<?php
// controllers/ProductoController.php
require_once __DIR__ . '/../includes/api_bootstrap.php'; // headers JSON+no-cache, TZ, charset, helpers

// === Utilidades de archivos ===
function rutaUploadsProductosFS(): string {
  // .../almacencrm/uploads/productos/
  return realpath(__DIR__ . '/../') . '/uploads/productos/';
}
function asegurarDirectorio(): string {
  $dir = rutaUploadsProductosFS();
  if (!is_dir($dir)) { @mkdir($dir, 0775, true); }
  return $dir;
}
function subirImagenProducto(?string &$errorMsg) {
  $errorMsg = null;
  if (empty($_FILES['imagen']) || $_FILES['imagen']['error'] !== UPLOAD_ERR_OK) {
    return null; // no subieron imagen
  }
  $tmp  = $_FILES['imagen']['tmp_name'];
  $orig = $_FILES['imagen']['name'] ?? '';
  $size = (int)($_FILES['imagen']['size'] ?? 0);

  $ext = strtolower(pathinfo($orig, PATHINFO_EXTENSION));
  $permitidas = ['jpg','jpeg','png','webp'];
  if (!in_array($ext, $permitidas, true)) { $errorMsg = 'Formato no permitido (jpg, jpeg, png, webp).'; return false; }
  if ($size > 2 * 1024 * 1024)           { $errorMsg = 'La imagen supera 2MB.';                         return false; }

  $dir = asegurarDirectorio();
  $nombre = 'prod_' . date('Ymd_His') . '_' . bin2hex(random_bytes(3)) . '.' . $ext;
  $destino = $dir . $nombre;
  if (!@move_uploaded_file($tmp, $destino)) {
    $errorMsg = 'No se pudo guardar la imagen en el servidor.';
    return false;
  }
  return $nombre; // nombre de archivo para DB
}
function borrarImagenFisica(?string $nombre): void {
  if (!$nombre) return;
  $ruta = rutaUploadsProductosFS() . $nombre;
  if (is_file($ruta)) { @unlink($ruta); }
}

// === Solo acepta POST ===
if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
  json_err('Método no permitido');
}

// =====================================
// ELIMINAR PRODUCTO
// =====================================
if (isset($_POST['accion']) && $_POST['accion'] === 'eliminar') {
  $id = as_nullable_int($_POST['id'] ?? null);
  if (!$id) json_err('ID de producto no válido');

  // Obtener imagen actual para borrarla del disco después
  $imgDel = null;
  if ($st = $conexion->prepare("SELECT imagen FROM productos WHERE id = ? LIMIT 1")) {
    $st->bind_param("i", $id);
    $st->execute();
    $st->store_result();
    $st->bind_result($imgDel);
    $st->fetch();
    $st->free_result();
    $st->close();
  }

  // Borrar QRs del producto (si existen)
  if ($st = $conexion->prepare("DELETE FROM codigos_qr WHERE producto_id = ?")) {
    $st->bind_param("i", $id);
    $st->execute();
    $st->close();
  }

  // Borrar producto
  $st = $conexion->prepare("DELETE FROM productos WHERE id = ?");
  $st->bind_param("i", $id);
  if (!$st->execute()) {
    $st->close();
    json_err('Error al eliminar el producto.');
  }
  $st->close();

  // Borrar imagen física
  if (!empty($imgDel)) borrarImagenFisica($imgDel);

  json_ok(['message' => 'Producto eliminado correctamente.']);
}

// =====================================
// CAPTURAR DATOS COMUNES
// =====================================
$id            = as_nullable_int($_POST['producto_id'] ?? null);
$codigo        = trim((string)($_POST['codigo'] ?? ''));
$nombre        = trim((string)($_POST['nombre'] ?? ''));
$tipo_id       = as_nullable_int($_POST['tipo_id'] ?? null);
$categoria_id  = as_nullable_int($_POST['categoria_id'] ?? null);
$medida_id     = as_nullable_int($_POST['medida_id'] ?? null);
$stock         = as_nullable_int($_POST['stock'] ?? 0)        ?? 0;
$stock_minimo  = as_nullable_int($_POST['stock_minimo'] ?? 0) ?? 0;

// Guardar DATE en formato YYYY-MM-DD; si no viene, NULL
$fecha_ingreso = trim((string)($_POST['fecha_ingreso'] ?? ''));
$fecha_ingreso = $fecha_ingreso !== '' ? $fecha_ingreso : null;

// Para conservar imagen en edición
$imagen_actual = trim((string)($_POST['imagen_actual'] ?? ''));

// QRs listados por textarea (uno por línea)
$codigos_qr_text   = (string)($_POST['codigos_qr'] ?? '');
$codigos_qr_array  = array_filter(array_map('trim', preg_split("/\r\n|\r|\n/", $codigos_qr_text)));

if ($codigo === '' || $nombre === '' || !$tipo_id || !$categoria_id || !$medida_id) {
  json_err('Completa todos los campos obligatorios.');
}

// Ajusta este ID según tu catálogo real
$TIPO_DEFECTUOSO = 1; // <-- Cambia si corresponde

// =====================================
// CASO: PRODUCTO DEFECTUOSO
// =====================================
if ((int)$tipo_id === (int)$TIPO_DEFECTUOSO) {
  if (!empty($codigos_qr_array)) {
    $sqlDef = "INSERT INTO productos_defectuosos (codigo_qr, fecha_registro, observacion, producto_id)
               VALUES (?, NOW(), ?, NULL)";
    $st = $conexion->prepare($sqlDef);
    foreach ($codigos_qr_array as $qr) {
      $obs = "Producto defectuoso registrado";
      $st->bind_param("ss", $qr, $obs);
      $st->execute();
    }
    $st->close();
  }
  json_ok(['message' => 'Producto defectuoso registrado correctamente.']);
}

// =====================================
// VALIDAR CÓDIGO ÚNICO
// =====================================
if ($id === null) {
  $st = $conexion->prepare("SELECT id FROM productos WHERE codigo = ? LIMIT 1");
  $st->bind_param("s", $codigo);
} else {
  $st = $conexion->prepare("SELECT id FROM productos WHERE codigo = ? AND id != ? LIMIT 1");
  $st->bind_param("si", $codigo, $id);
}
$st->execute();
$st->store_result();
if ($st->num_rows > 0) { $st->free_result(); $st->close(); json_err('El código general ya está en uso.'); }
$st->free_result();
$st->close();

// =====================================
// SUBIR IMAGEN (si viene)
// =====================================
$errorImg = null;
$imagenSubida = subirImagenProducto($errorImg);
if ($imagenSubida === false) {
  json_err($errorImg ?: 'Error al subir imagen');
}
// decidir qué imagen guardar
$imagenAGuardar = $imagenSubida ?: ($imagen_actual ?: null);

// =====================================
// INSERTAR PRODUCTO
// =====================================
if ($id === null) {
  $sql = "INSERT INTO productos (codigo, nombre, imagen, tipo_id, categoria_id, medida_id, stock, stock_minimo, fecha_ingreso)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
  $st = $conexion->prepare($sql);
  $st->bind_param(
    "sssiiiiis", // 3s + 5i + 1s
    $codigo, $nombre, $imagenAGuardar, $tipo_id, $categoria_id, $medida_id, $stock, $stock_minimo, $fecha_ingreso
  );
  if (!$st->execute()) {
    $st->close();
    if ($imagenSubida) borrarImagenFisica($imagenSubida); // no dejar basura
    json_err('Error al registrar el producto.');
  }
  $producto_id = $st->insert_id;
  $st->close();

  // Guardar QRs (si enviaron)
  if (!empty($codigos_qr_array)) {
    $stQR = $conexion->prepare("INSERT INTO codigos_qr (producto_id, codigo_qr) VALUES (?, ?)");
    foreach ($codigos_qr_array as $qr) {
      $stQR->bind_param("is", $producto_id, $qr);
      $stQR->execute();
    }
    $stQR->close();
  }

  json_ok(['message' => 'Producto registrado con éxito.']);

// =====================================
// ACTUALIZAR PRODUCTO
// =====================================
} else {
  // Si subieron imagen nueva, borrar la anterior del disco
  if ($imagenSubida && $imagen_actual) {
    borrarImagenFisica($imagen_actual);
  }

  $sql = "UPDATE productos
          SET codigo=?, nombre=?, imagen=?, tipo_id=?, categoria_id=?, medida_id=?, stock=?, stock_minimo=?, fecha_ingreso=?
          WHERE id=?";
  $st = $conexion->prepare($sql);
  $st->bind_param(
    "sssiiiiisi", // 3s, 5i, 1s, 1i
    $codigo, $nombre, $imagenAGuardar, $tipo_id, $categoria_id, $medida_id, $stock, $stock_minimo, $fecha_ingreso, $id
  );
  if (!$st->execute()) {
    $st->close();
    if ($imagenSubida) borrarImagenFisica($imagenSubida); // revertir archivo nuevo
    json_err('Error al actualizar el producto.');
  }
  $st->close();

  // Reemplazar QRs (si se envió textarea)
  if ($st = $conexion->prepare("DELETE FROM codigos_qr WHERE producto_id = ?")) {
    $st->bind_param("i", $id);
    $st->execute();
    $st->close();
  }
  if (!empty($codigos_qr_array)) {
    $stQR = $conexion->prepare("INSERT INTO codigos_qr (producto_id, codigo_qr) VALUES (?, ?)");
    foreach ($codigos_qr_array as $qr) {
      $stQR->bind_param("is", $id, $qr);
      $stQR->execute();
    }
    $stQR->close();
  }

  json_ok(['message' => 'Producto actualizado con éxito.']);
}
